//
//  GFMView.m
//
//  Created by Colin Walsh <colin@celsiusgs.com> on 10-07-06.
//  Copyright 2010 Celsius Game Studios.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write to the Free Software Foundation, Inc.,
//  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


#include <stdio.h>
#import <AppKit/AppKit.h>
#import "GFMView.h"
#import "bitmapcontext.h"

@implementation GFMView
-(id)initWithFrame:(NSRect)frameRect {
	bitmapRef = nil;
	bitmapData = NULL;
	drawFont = [NSFont fontWithName:@"Helvetica" size:26];
	drawChar = ' ';
	charWidth = 0;
	charHeight = 0;
	ascender = 0;
	descender = 0;
	doPreview = true;
	if ((self = [super initWithFrame:frameRect]) == nil)
	{
		return self;
	}
	return self;
}
//Specify the NSFont that is being drawn with
-(void)setDrawFont:(NSFont *)font{
	drawFont = font;
}
//Draw character in the bottom left hand corner of the view
-(void)doDrawChar:(char)c {
	
	drawChar = c;
	[self setNeedsDisplay:YES];
}
//Returns the bitmap that represents the captured character
-(unsigned char *)getBitmapData
{
	return bitmapData;
}
//Gets the CGContextRef that points to the captured character
-(CGContextRef)getBitmapRef {
	return bitmapRef;
}
//Gets the computed width of the captured character
//This is either the advance or the width of the bounding box
//that completely encompasses the character, whichever is larger
-(int)getCharWidth {
	return charWidth;
}
//Gets the height of the captured character in pixels, this is generally the
//sum of the absolute values of the ascender and descender
-(int)getCharHeight {
	return charHeight;
}
//Gets the height of the font in pixels, which for our purposes is the sum
//of the absolute values of the ascender and descender
-(int)getFontHeight {
	int height = abs([drawFont ascender]) + abs([drawFont descender]);
	return height;
}
//Get the ascender as computed at the last capture
-(float)getAscender {
	return ascender;
}
//Get the descender as computed at the last capture
-(float)getDescender {
	return descender;
}
//If this is set, display a short preview string in the view
-(void)setDoPreview:(bool)prev {
	doPreview = prev;
}
//Compute the width of the glyph represented by c
//this is either the advance or the width of the bounding box that
//encloses the glyph, whichever is larger
-(int)getCharWidth:(char)c {
	NSString *charString = [NSString stringWithFormat:@"%c", c];
	
	NSGlyph g;
	
	NSTextStorage *ts = [[NSTextStorage alloc] initWithString:charString];
	[ts setFont:drawFont];
	NSLayoutManager *lm = [[NSLayoutManager alloc] init];
	NSTextContainer *tc = [[NSTextContainer alloc] init];
	
	[lm addTextContainer:tc];
	[tc release]; // lm retains tc
	[ts addLayoutManager:lm];
	[lm release]; // ts retains lm
	g = [lm glyphAtIndex:0];
	[ts release];
	
	NSBezierPath *path = [NSBezierPath bezierPath];
	NSPoint startPoint = { 0, 0 };
	[path moveToPoint: startPoint];
	[path appendBezierPathWithGlyph:g inFont:drawFont];
	NSRect pathBounds = [path bounds];
	int width = pathBounds.size.width;//[drawFont advancementForGlyph:g].width;
	if([drawFont advancementForGlyph:g].width > pathBounds.size.width)
	{
		width = (int)rint([drawFont advancementForGlyph:g].width);
	}else{
		width = (int)rint(pathBounds.size.width);
	}
	return width;
}
//Drawing method for the view
-(void)drawRect:(NSRect)theRect {
	
	char tempChar[2];
	tempChar[0] = drawChar;
	tempChar[1] = 0;
	
	[[NSColor whiteColor] set];
	NSRectFill( theRect );
	
	NSString *charString = [NSString stringWithFormat:@"%c", drawChar];
	NSDictionary *attributes = [NSDictionary dictionaryWithObjectsAndKeys:drawFont, NSFontAttributeName,[NSColor blackColor], NSForegroundColorAttributeName, nil];
	
	NSAttributedString *currentText;
	if(!doPreview)
	{
		currentText=[[NSAttributedString alloc] initWithString:charString attributes: attributes];
	}else{
		currentText=[[NSAttributedString alloc] initWithString:@"Sample 1234567890" attributes: attributes];
	}
		
	//Here be dragons (This code is required to figure out the true width of the character!)
	NSGlyph g;
	
	NSTextStorage *ts = [[NSTextStorage alloc] initWithString:charString];
	[ts setFont:drawFont];
	NSLayoutManager *lm = [[NSLayoutManager alloc] init];
	NSTextContainer *tc = [[NSTextContainer alloc] init];
	
	[lm addTextContainer:tc];
	[tc release]; // lm retains tc
	[ts addLayoutManager:lm];
	[lm release]; // ts retains lm
	g = [lm glyphAtIndex:0];
	[ts release];
	
	NSBezierPath *path = [NSBezierPath bezierPath];
	NSPoint startPoint = { 0, 0 };
	[path moveToPoint: startPoint];
	[path appendBezierPathWithGlyph:g inFont:drawFont];
	NSRect pathBounds = [path bounds];
	int width = pathBounds.size.width;
	
	//Dragons end here
	
	//This is some number fudging to prevent some fonts which don't appear in the right place
	int vOff = (abs([drawFont ascender]) + abs([drawFont descender])) - [drawFont pointSize];
	int hOff = -pathBounds.origin.x;
	if(vOff > 0)vOff = 0;
	if(doPreview)
	{
		vOff = hOff = 0;
	}
	[currentText drawAtPoint:NSMakePoint(hOff, vOff)];
	
	if([drawFont advancementForGlyph:g].width > pathBounds.size.width)
	{
		width = (int)rint([drawFont advancementForGlyph:g].width);
	}else{
		width = (int)rint(pathBounds.size.width);
	}
	int height = abs([drawFont ascender]) + abs([drawFont descender]);
	ascender = [drawFont ascender];
	descender = [drawFont descender];
	charWidth = width;
	charHeight = height;
	
	//This code captures the contents of the view to a bitmap
	[self lockFocus];
	NSBitmapImageRep *bits;
	bits = [[NSBitmapImageRep alloc] initWithFocusedViewRect: [self bounds]];
	[self unlockFocus];
	
	CGContextRef bitmapContext = createBitmapContext(width, height);
	
	CGContextSetRGBFillColor (bitmapContext, 0, 0, 0, 0);
	CGContextFillRect(bitmapContext, CGRectMake(0, 0, width, height));

	CGContextDrawImage(bitmapContext, CGRectMake(0, 0, [self bounds].size.width, [self bounds].size.height), [bits CGImage]);
	
	bitmapData = (unsigned char *)CGBitmapContextGetData(bitmapContext); 
	bitmapRef = bitmapContext;
}
@end
